within Buildings.Fluid.Sensors;
model SensibleEnthalpyFlowRate
  "Ideal enthalphy flow rate sensor that outputs the sensible enthalpy flow rate only"
  extends Buildings.Fluid.Sensors.BaseClasses.PartialDynamicFlowSensor(tau=0);
  extends Buildings.Fluid.BaseClasses.IndexWater;
  extends Modelica.Icons.RotationalSensor;

  Modelica.Blocks.Interfaces.RealOutput H_flow(unit="W")
    "Sensible enthalpy flow rate, positive if from port_a to port_b"
    annotation (Placement(transformation(
        origin={0,110},
        extent={{-10,-10},{10,10}},
        rotation=90)));
  parameter Modelica.SIunits.SpecificEnthalpy h_out_start=
    Medium.enthalpyOfNonCondensingGas(
      Medium.temperature(Medium.setState_phX(
        Medium.p_default, Medium.T_default, Medium.X_default)))
    "<html>Initial or guess value of measured specific <b>sensible</b> enthalpy</html>"
    annotation (Dialog(group="Initialization"));
  Modelica.SIunits.SpecificEnthalpy hMed_out(start=h_out_start)
    "Medium sensible enthalpy to which the sensor is exposed";
  Modelica.SIunits.SpecificEnthalpy h_out(start=h_out_start)
    "Medium sensible enthalpy that is used to compute the enthalpy flow rate";
protected
  Medium.MassFraction XiActual[Medium.nXi]
    "Medium mass fraction to which sensor is exposed to";
  Medium.SpecificEnthalpy hActual
    "Medium enthalpy to which sensor is exposed to";
  Medium.ThermodynamicState sta "Medium state to which sensor is exposed to";
initial equation
 // Compute initial state
 if dynamic then
    if initType == Modelica.Blocks.Types.Init.SteadyState then
      der(h_out) = 0;
    elseif initType == Modelica.Blocks.Types.Init.InitialState or
           initType == Modelica.Blocks.Types.Init.InitialOutput then
      h_out = h_out_start;
    end if;
  end if;
equation
  if allowFlowReversal then
     XiActual = Modelica.Fluid.Utilities.regStep(port_a.m_flow,
                 port_b.Xi_outflow,
                 port_a.Xi_outflow, m_flow_small);
     hActual = Modelica.Fluid.Utilities.regStep(port_a.m_flow,
                 port_b.h_outflow,
                 port_a.h_outflow, m_flow_small);
  else
     XiActual = port_b.Xi_outflow;
     hActual = port_b.h_outflow;
  end if;
  // Specific enthalpy measured by sensor
  sta = Medium.setState_phX(port_a.p, hActual, XiActual);
  hMed_out = (1-XiActual[i_w]) * Medium.enthalpyOfNonCondensingGas(
      Medium.temperature(sta));
  if dynamic then
    der(h_out) = (hMed_out-h_out)*k/tau;
  else
    h_out = hMed_out;
  end if;
  // Sensor output signal
  H_flow = port_a.m_flow * h_out;
annotation (defaultComponentName="senEntFlo",
  Diagram(coordinateSystem(preserveAspectRatio=true,  extent={{-100,-100},{100,
            100}})),
  Icon(graphics={
        Ellipse(
          extent={{-70,70},{70,-70}},
          lineColor={0,0,0},
          fillColor={170,213,255},
          fillPattern=FillPattern.Solid),
        Line(points={{-100,0},{-70,0}}, color={0,128,255}),
        Line(points={{70,0},{100,0}}, color={0,128,255}),
        Line(points={{0,100},{0,70}}, color={0,0,127}),
        Text(
          extent={{180,151},{20,99}},
          lineColor={0,0,0},
          textString="HS_flow"),
        Polygon(
          points={{-0.48,31.6},{18,26},{18,57.2},{-0.48,31.6}},
          lineColor={0,0,0},
          fillColor={0,0,0},
          fillPattern=FillPattern.Solid),
        Line(points={{0,0},{9.02,28.6}}, color={0,0,0}),
        Ellipse(
          extent={{-5,5},{5,-5}},
          lineColor={0,0,0},
          fillColor={0,0,0},
          fillPattern=FillPattern.Solid),
        Line(points={{37.6,13.7},{65.8,23.9}}, color={0,0,0}),
        Line(points={{22.9,32.8},{40.2,57.3}}, color={0,0,0}),
        Line(points={{0,70},{0,40}}, color={0,0,0}),
        Line(points={{-22.9,32.8},{-40.2,57.3}}, color={0,0,0}),
        Line(points={{-37.6,13.7},{-65.8,23.9}}, color={0,0,0})}),
  Documentation(info="<html>
<p>
This component monitors the <i>sensible</i> enthalphy flow rate of the medium in the flow
between fluid ports. In particular, if the total enthalpy flow rate is
<p align=\"center\" style=\"font-style:italic;\">
  H&#775;<sub>tot</sub> = H&#775;<sub>sen</sub> + H&#775;<sub>lat</sub>,
</p>
where 
<i>H&#775;<sub>sen</sub> = m&#775; (1-X<sub>w</sub>) c<sub>p,air</sub></i>, 
then this sensor outputs <i>H&#775; = H&#775;<sub>sen</sub></i>. 
</p>
<p>
If the parameter <code>tau</code> is non-zero, then the measured
specific sensible enthalpy <i>h<sub>out</sub></i> that is used to 
compute the sensible enthalpy flow rate 
<i>H&#775;<sub>sen</sub> = m&#775; h<sub>out</sub></i> 
is computed using a first order differential equation. 
See <a href=\"modelica://Buildings.Fluid.Sensors.UsersGuide\">
Buildings.Fluid.Sensors.UsersGuide</a> for an explanation.
</p>
<p>
For a sensor that measures 
<i>H&#775;<sub>tot</sub></i>, use
<a href=\"modelica://Buildings.Fluid.Sensors.EnthalpyFlowRate\">
Buildings.Fluid.Sensors.EnthalpyFlowRate</a>.<br>
For a sensor that measures 
<i>H&#775;<sub>lat</sub></i>, use
<a href=\"modelica://Buildings.Fluid.Sensors.LatentEnthalpyFlowRate\">
Buildings.Fluid.Sensors.LatentEnthalpyFlowRate</a>.
<p>
The sensor is ideal, i.e., it does not influence the fluid.
The sensor can only be used with medium models that implement the function
<code>enthalpyOfNonCondensingGas(state)</code>.
</p>
</html>
", revisions="<html>
<ul>
<li>
December 18, 2012, by Michael Wetter:<br>
Moved computation of <code>i_w</code> to new base class
<a href=\"modelica://Buildings.Fluid.BaseClasses.IndexWater\">
Buildings.Fluid.BaseClasses.IndexWater</a>.
The value of this parameter is now assigned dynamically and does not require to be specified
by the user.
</li>
<li>
November 3, 2011, by Michael Wetter:<br>
Moved <code>der(h_out) := 0;</code> from the initial algorithm section to 
the initial equation section
as this assignment does not conform to the Modelica specification.
</li>
<li>
August 10, 2011 by Michael Wetter:<br>
Added parameter <code>i_w</code> and an assert statement to
make sure it is set correctly. Without this change, Dymola
cannot differentiate the model when reducing the index of the DAE.
</li>
<li>
June 3, 2011 by Michael Wetter:<br>
Revised implementation to add dynamics in such a way that 
the time constant increases as the mass flow rate tends to zero.
This can improve the numerics.
</li>
<li>
February 22, by Michael Wetter:<br>
Improved code that searches for index of 'water' in the medium model.
</li>
<li>
September 9, 2009 by Michael Wetter:<br>
First implementation.
Implementation is based on enthalpy sensor of <code>Modelica.Fluid</code>.
</li>
</ul>
</html>"));
end SensibleEnthalpyFlowRate;
